% Script file fault_slip_plot_regularizations
%
% Produces Figure for Appendix B. Not used in computation.

clear variables

a = 75; % see Figure 1, m
alpha = 0.9; % Biot's coefficient, -
b = 150; % see Figure 1, m
D_center = 3500; % reservoir depth @ y=0, m
g = 9.81; % acceleration of gravity, m/s^2
G = 6500e6; % shear modulus, Pa
kappa = 0; % cohesion, -
K_eff = 0.5; % initial effective vertical/horizontal stress ratio, -
mu_st = 0.52; % static friction coefficient, -
nu = 0.15; % Poisson's ratio, -
p_0_center = 35e6; % initial pressure @ y=0, Pa
p_inc = -25e6; % incremental pressure, Pa
phi = 0.15; % porosity, -
rho_f = 1020; % fluid density, kg/m^3
rho_s = 2650; % solid density, kg/m^3
sigma_v_0 = 0; % initial vertical stress @ y=0, Pa. If set equal to zero, the stress is ...
% computed from the weigth of the overburden; see function fault_slip_sigma.m.
theta = 70*pi/180; % dip angle, rad

% Data for plots:
y_plot_max =  250; % m
y_plot_min = -250; % m
blue = [0/255 128/255 255/255];
gray = [235/255 235/255 235/255];
green = [0/255 204/255 0/255];
orange = [255/255 185/255 0/255];

% Set domain:
eta_reg_1 = 0.10; % regularization parameter; value used for examples in paper
eta_reg_2 = 2; % regularization parameter; exaggerated value
y = linspace(y_plot_min,y_plot_max,5000);

% Compute auxiliary variables:
C = ((1-2*nu)*alpha*p_inc) / (2*pi*(1-nu)); % scaling parameter, N/m^2

% Compute gradient:
p_0 = p_0_center - rho_f * g * y; % p_0_center is the initial pressure @ y=0

% Compute initial stresses:
if sigma_v_0 == 0 % compute sigma_v_0 from overburden weigth
    sigma_v_0 = ( (1-phi)*rho_s + phi*rho_f ) * g * (y-D_center); % initial vertical stress
end
sigma_v_eff_0 = sigma_v_0 + alpha * p_0; % initial effective vertical stress
sigma_h_eff_0 = K_eff * sigma_v_eff_0; % initial effective horizontal stress
sigma_h_0 = sigma_h_eff_0 - alpha * p_0; % initial horizontal stress
sigma_norm_0  = sigma_h_0 * (sin(theta))^2 + sigma_v_0 * (cos(theta))^2; % initial normal str.
sigma_shear_0 = - (sigma_v_0 - sigma_h_0) * sin(theta) * cos(theta); % initial shear stress

% Compute incremental stresses:
% % % sigma_xx_1 = ...
% % %      0      * ( (y <= -b) + (b <= y) ) + ...
% % %     -pi*C   * ( ( (-b < y) .* (y <= -a) ) + ( (a <= y) .* (y < b) ) ) + ... 
% % %     -2*pi*C * (-a < y) .* (y < a);     
sigma_xx_1 =   -C * ( atan2( (a+b)*eta_reg_1 , eta_reg_1.^2+(y-b).*(y+a) ) + ...
                      atan2( (a+b)*eta_reg_1 , eta_reg_1.^2+(y-a).*(y+b) ) );                
sigma_xx_2 =   -C * ( atan2( (a+b)*eta_reg_2 , eta_reg_2.^2+(y-b).*(y+a) ) + ...
                      atan2( (a+b)*eta_reg_2 , eta_reg_2.^2+(y-a).*(y+b) ) );                

% % % sigma_xy_1 = (C/2) * log( ((y-a).^2.*(y+a).^2)./((y-b).^2.*(y+b).^2) );
sigma_xy_1 = (C/2) * log( ( (eta_reg_1^2+(y-a).^2).*(eta_reg_1^2+(y+a).^2) ) ./ ...
                          ( (eta_reg_1^2+(y-b).^2).*(eta_reg_1^2+(y+b).^2) ) );
sigma_xy_2 = (C/2) * log( ( (eta_reg_2^2+(y-a).^2).*(eta_reg_2^2+(y+a).^2) ) ./ ...
                          ( (eta_reg_2^2+(y-b).^2).*(eta_reg_2^2+(y+b).^2) ) );

shift_1 = alpha*p_inc/pi * atan2((a+b)*eta_reg_1,eta_reg_1^2+(y-b).*(y+b));
shift_2 = alpha*p_inc/pi * atan2((a+b)*eta_reg_2,eta_reg_2^2+(y-b).*(y+b));

% % % sigma_xx_eff_1 = sigma_xx_1 + alpha*p_inc * (-b < y) .* (y < b);
sigma_xx_eff_1 = sigma_xx_2 + shift_1;              
sigma_xx_eff_2 = sigma_xx_2 + shift_2;              
                      
sigma_norm_1 =  (-sigma_xy_1 * sin(theta)*cos(theta) + sigma_xx_1 * sin(theta)^2 ); 
sigma_norm_2 =  (-sigma_xy_2 * sin(theta)*cos(theta) + sigma_xx_2 * sin(theta)^2 ); 

sigma_shear_1 = (sigma_xy_1 * sin(theta)^2 + sigma_xx_1 * sin(theta)*cos(theta) ); 
sigma_shear_2 = (sigma_xy_2 * sin(theta)^2 + sigma_xx_2 * sin(theta)*cos(theta) ); 

% Compute effective stresses:
sigma_norm_eff_0 = sigma_norm_0 + alpha*p_0;
% % % sigma_norm_eff_1 = sigma_norm_1 + alpha*p_inc * (-b < y) .* (y < b);
sigma_norm_eff_1 = sigma_norm_2 + shift_1;
sigma_norm_eff_2 = sigma_norm_2 + shift_2;

% Compute combined quantities:
p_comb = p_0 + p_inc * (-b < y) .* (y < b);
sigma_norm_comb_1 = sigma_norm_0 + sigma_norm_1; 
sigma_norm_comb_2 = sigma_norm_0 + sigma_norm_2; 
sigma_norm_eff_comb_1 = sigma_norm_eff_0 + sigma_norm_eff_1;
sigma_norm_eff_comb_2 = sigma_norm_eff_0 + sigma_norm_eff_2;
sigma_shear_comb_1 = sigma_shear_0 + sigma_shear_1;
sigma_shear_comb_2 = sigma_shear_0 + sigma_shear_2;

% Compute right-hand slip boundary and Coulomb stresses:
sigma_slip_1 = kappa-mu_st*sigma_norm_eff_comb_1; % slip boundary; reservoir-only incr. pr.
sigma_slip_2 = kappa-mu_st*sigma_norm_eff_comb_2; % slip boundary; reservoir-only incr. pr.
sigma_C_1 = sigma_shear_comb_1 - sigma_slip_1; % Coulomb stress; reservoir-only incremental pressure
sigma_C_2 = sigma_shear_comb_2 - sigma_slip_2; % Coulomb stress; reservoir-only incremental pressure

% Slip-weaking friction:
mu_st = 0.52;
mu_dyn = 0.2;
delta_c = 0.02;
eps = 0.10 * delta_c; 

x_min = 0.0;
x_max = 0.03;
y_min = 0.0;
y_max = 0.6;

delta = linspace(0,1.5*delta_c,200); % parameter
a1 = mu_st;
b1 = 0;
c1 = (mu_st-mu_dyn)/(2*eps^2-2*delta_c*eps);
a2 = (eps*(mu_st+mu_dyn)-2*delta_c*mu_st)/(2*eps-2*delta_c);
b2 = (mu_st-mu_dyn)/(eps-delta_c);
a3 = (delta_c^2*(mu_dyn-mu_st)+2*eps^2*mu_dyn-2*delta_c*eps*mu_dyn)/(2*eps^2-2*delta_c*eps);
b3 = (delta_c*(mu_st-mu_dyn))/(eps^2-delta_c*eps);
c3 = -(mu_st-mu_dyn)/(2*eps^2-2*delta_c*eps);
mu = (a1+b1*delta+c1*delta.^2) .* (delta < eps) + ...
     (a2+b2*delta)             .* (eps <= delta) .* (delta < delta_c-eps) + ...
     (a3+b3*delta+c3*delta.^2) .* (delta_c-eps <= delta) .* (delta < delta_c) + ...
     mu_dyn                    .* (delta_c <= delta);
mu_1 = (a1+b1*eps+c1*eps.^2); 
mu_3 = (a3+b3*(delta_c-eps)+c3*(delta_c-eps).^2);

% Create plots:
figure 
subplot(1,2,1)
hold on
f_stretch = 3; % stretch factor for plotting of fault blocks
x_patch_left = [-21.9 -21.9 -19+((b-a)/tan(theta)/f_stretch) -19];
y_patch_left = [-150 75 75 -150];
patch(x_patch_left,y_patch_left,gray,'LineStyle','none')
x_patch_right = [5-((b-a)/tan(theta)/f_stretch) 5 7.9 7.9];
y_patch_right = [-75 150 150 -75];
patch(x_patch_right,y_patch_right,gray,'LineStyle','none')
x_plot_min = -22;
x_plot_max = 8;
line([x_plot_min x_plot_max],[150 150],'LineStyle',':','Color','k','LineWidth',1)
line([x_plot_min x_plot_max],[75 75],'LineStyle',':','Color','k','LineWidth',1)
line([x_plot_min x_plot_max],[-75 -75],'LineStyle',':','Color','k','LineWidth',1)
line([x_plot_min x_plot_max],[-150 -150],'LineStyle',':','Color','k','LineWidth',1)
line([x_plot_min x_plot_max],[0 0],'LineStyle','-','Color','k','LineWidth',0.5)
line([0 0],[y_plot_min y_plot_max],'LineStyle','-','Color','k','LineWidth',0.5)
plot(sigma_C_1/1e6,y,'-r','LineWidth',1.5)
plot(sigma_C_2/1e6,y,'LineStyle','-','Color',[0/255 128/255 255/255],'LineWidth',1.5) % blue
hold off
box on
axis([x_plot_min x_plot_max y_plot_min y_plot_max])
xlabel('$\Sigma_C$ (MPa)','Interpreter','latex','FontSize',11)
ylabel('$\it y$ (m)','Interpreter','latex','FontSize',11)

% subplot(1,2,2)
% hold on
% line([x_min x_max],[mu_st mu_st],'LineStyle',':','Color','k','LineWidth',0.5)
% line([x_min x_max],[mu_dyn mu_dyn],'LineStyle',':','Color','k','LineWidth',0.5)
% line([eps eps],[y_min y_max],'LineStyle',':','Color','k','LineWidth',0.5)
% line([delta_c-eps delta_c-eps],[y_min y_max],'LineStyle',':','Color','k','LineWidth',0.5)
% line([delta_c delta_c],[y_min y_max],'LineStyle',':','Color','k','LineWidth',0.5)
% %line([0 0.02],[mu_st mu_dyn],'LineStyle','-','Color','r','LineWidth',1.5)
% plot(delta,mu,'Color',[0/255 128/255 255/255],'LineWidth',1.5)
% hold off
% xlabel('$\it \delta \rm(m)$','Interpreter','latex','FontSize',11)
% ylabel('$\it \mu \rm(-)$','Interpreter','latex','FontSize',11)
% text(0.0305,mu_dyn,'$\it \mu_{dyn}$','Interpreter','latex','FontSize',11)
% text(0.0305,mu_st,'$\it \mu_{st}$','Interpreter','latex','FontSize',11)
% text(eps-0.0003,0.62,'$\it \epsilon$','Interpreter','latex','FontSize',11)
% text(delta_c-eps-0.0050,0.62,'$\it \delta_c\!-\!\epsilon$','Interpreter','latex','FontSize',11)
% text(delta_c-0.0005,0.62,'$\it \delta_c$','Interpreter','latex','FontSize',11)
% axis([x_min x_max y_min y_max]);
% box on
